/* Copyright 2018 Canaan Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <stdio.h>
#include <string.h>

#include "board_config.h"
#include "dvp.h"
#include "fpioa.h"
#include "gc0328.h"
#include "lcd.h"
#include "nt35310.h"
#include "plic.h"
#include "sysctl.h"
#include "uarths.h"
#include "unistd.h"

uint32_t g_lcd_gram0[38400] __attribute__((aligned(64)));
uint32_t g_lcd_gram1[38400] __attribute__((aligned(64)));

volatile uint8_t g_dvp_finish_flag;
volatile uint8_t g_ram_mux;
static uint8_t count = 0;
static int on_irq_dvp(void *ctx) {
  if (dvp_get_interrupt(DVP_STS_FRAME_FINISH)) {
    /* switch gram */
    dvp_set_display_addr(g_ram_mux ? (uint32_t)g_lcd_gram0
                                   : (uint32_t)g_lcd_gram1);

    dvp_clear_interrupt(DVP_STS_FRAME_FINISH);
    g_dvp_finish_flag = 1;

    if (count++ > 10) {
      io_set_650();
      gc0328_output_dis();
      io_set_850();
      gc0328_output_en();
    }
    if (count > 20) {
      count = 0;
      io_set_850();
      gc0328_output_dis();
      io_set_650();
      gc0328_output_en();
    }

  } else {
    if (g_dvp_finish_flag == 0) dvp_start_convert();
    dvp_clear_interrupt(DVP_STS_FRAME_START);
  }

  return 0;
}

static void io_mux_init(void) {
  /* Init DVP IO map and function settings */
  fpioa_set_function(42, FUNC_CMOS_RST);
  fpioa_set_function(13, FUNC_CMOS_PWDN);
  fpioa_set_function(46, FUNC_CMOS_XCLK);
  fpioa_set_function(43, FUNC_CMOS_VSYNC);
  fpioa_set_function(45, FUNC_CMOS_HREF);
  fpioa_set_function(47, FUNC_CMOS_PCLK);
  fpioa_set_function(41, FUNC_SCCB_SCLK);  // sclk
  fpioa_set_function(40, FUNC_SCCB_SDA);   // sda 650
  fpioa_set_function(44, FUNC_GPIO2);      // sda 850

  /* Init SPI IO map and function settings */
  fpioa_set_function(38, FUNC_GPIOHS0 + DCX_GPIONUM);
  fpioa_set_function(36, FUNC_SPI0_SS3);
  fpioa_set_function(39, FUNC_SPI0_SCLK);
  fpioa_set_function(37, FUNC_GPIOHS0 + RST_GPIONUM);

  sysctl_set_spi0_dvp_data(1);
}

static void io_set_power(void) {
  /* Set dvp and spi pin to 1.8V */
  sysctl_set_power_mode(SYSCTL_POWER_BANK6, SYSCTL_POWER_V18);
  sysctl_set_power_mode(SYSCTL_POWER_BANK7, SYSCTL_POWER_V18);
}

void io_set_650(void) {
  uint8_t i = 0;
  i += fpioa_set_function(44, FUNC_GPIO2);
  i += fpioa_set_function(40, FUNC_SCCB_SDA);
}

void io_set_850(void) {
  uint8_t i = 0;
  i += fpioa_set_function(40, FUNC_GPIO2);
  i += fpioa_set_function(44, FUNC_SCCB_SDA);
}

int main(void) {
  /* Set CPU and dvp clk */
  sysctl_pll_set_freq(SYSCTL_PLL0, 700000000UL);
  sysctl_pll_set_freq(SYSCTL_PLL1, 160000000UL);
  sysctl_pll_set_freq(SYSCTL_PLL2, 45158400UL);
  uarths_init();

  io_mux_init();
  io_set_power();
  plic_init();

  /* LCD init */
  printf("LCD init\n");
  lcd_init();

  /* DVP init */
  printf("DVP init\n");

  dvp_init(8);
  dvp_set_xclk_rate(24000000);
  dvp_enable_burst();
  dvp_set_output_enable(0, 1);
  dvp_set_output_enable(1, 1);
  dvp_set_image_format(DVP_CFG_RGB_FORMAT);
  dvp_set_image_size(320, 240);
  io_set_850();
  gc0328_init();

  io_set_650();
  gc0328_init();
  gc0328_output_en();
  dvp_set_ai_addr((uint32_t)0x40600000, (uint32_t)0x40612C00,
                  (uint32_t)0x40625800);
  dvp_set_display_addr((uint32_t)g_lcd_gram0);
  dvp_config_interrupt(DVP_CFG_START_INT_ENABLE | DVP_CFG_FINISH_INT_ENABLE, 0);
  dvp_disable_auto();

  /* DVP interrupt config */
  printf("DVP interrupt config\n");
  plic_set_priority(IRQN_DVP_INTERRUPT, 1);
  plic_irq_register(IRQN_DVP_INTERRUPT, on_irq_dvp, NULL);
  plic_irq_enable(IRQN_DVP_INTERRUPT);

  /* enable global interrupt */
  sysctl_enable_irq();

  /* system start */
  printf("system start\n");
  g_ram_mux = 0;
  g_dvp_finish_flag = 0;
  dvp_clear_interrupt(DVP_STS_FRAME_START | DVP_STS_FRAME_FINISH);
  dvp_config_interrupt(DVP_CFG_START_INT_ENABLE | DVP_CFG_FINISH_INT_ENABLE, 1);

  while (1) {
    /* ai cal finish*/
    while (g_dvp_finish_flag == 0)
      ;
    g_dvp_finish_flag = 0;
    /* display pic*/
    g_ram_mux ^= 0x01;
    lcd_draw_picture(0, 0, 320, 240, g_ram_mux ? g_lcd_gram0 : g_lcd_gram1);
  }

  return 0;
}
